%--------------------------------------------------------------------------
% Metformin paper compartmental model for placenta perfusion experiments
%
%                      Bram Sengers 2021
%--------------------------------------------------------------------------
clear
close all;

% --- Initialize input parameters -----------------------------------------
param.perfcase=2;

switch param.perfcase
    case 1 % Bidirectional experiment
        %param.expdat='expdata587';  % separate m-file with the experimental data for each placenta, used for plotting and fitting
        %param.expdat='expdata604';
        %param.expdat='expdata608';;
        param.expdat='expdata_average_Bidir_exp'; % average data
    case 2 % OCT3 experiment
        % param.expdat='expdata627_crea'
        % param.expdat='expdata633_crea'
        % param.expdat='expdata641_crea'
        % param.expdat='expdata657_crea'
        param.expdat='expdata_average_OCT3_exp_crea'; % average creatinine data
end
plotdata = true;        % set to true for plotting experimental data, or else set to false if no data is available
str = param.expdat;
eval(str);              % load experimental results stored in separate file


% Fit parameters to data or calculate results with given parameters
param.fitcase=1;        % 1=diffusion only, 2=diffusion+OCT3, 3=transporters only SERT, OCT3 and MVM efflux, 4=everything,diffusion and all transporters 
fitpar = false;         % use fitpar=true to estimate parameters and fitpar=false to calculate with given parameter values


% --- Flow rates ---
param.Fm=14e-3;                   % [dm^3/min]
%param.Ff=6e-3;                   % [dm^3/min] nominal fetal flow, not used now
param.Fflist=Fetalflowflist*1e-3; % [dm^3/min] measured fetal flow during the experiment

param.A=1;              % [dm^2] set to 1 if exchange area is lumped with permeabilities 
Vtot = weight*1e-3;     % [dm^3] volume based on cotelydon weight in grams

% --- Relative volume fractions compartments --- 
mfrac=0.34;             % [-] intervillous space          
ffrac=0.074;            % [-] fetal capillaries
sfrac=0.15;             % [-] syncytiotrophoblast -> increase this as now we are talking about tissue in general!


param.V1=mfrac*Vtot;          % [dm^3] maternal volume
param.V2=sfrac*Vtot;          % [dm^3] tissue volume
param.V3=ffrac*Vtot;          % [dm^3] fetal volume


% --- Paracellular diffusion:
param.PdifA=0.00046438;      % [dm min^-1] diffusive permeability solute A: creatinine

param.PdifB=param.PdifA ;    % [dm min^-1] diffusive permeability solute B: unlabelled metformin
param.PdifC=param.PdifA ;    % [dm min^-1] diffusive permeability solute C: other competitive inhibitor


% --- Transporter kinetic parameters:   

% General parameters
z = 1;% electrical charge of the charged solute (including sign for positive or negative charge)
F = 9.6485e4; % [C mol^-1] Faraday constant
R = 8.314; % [VCK^-1 mol^-1] Gas constant
T = 310; % [K] Absolute temperature
deltaPsi =-18e-3;% [V] membrane potential difference inside with respect to outside cell
beta = 0.5; % electrical bias

% --- Uptake on MVM via SERT:
param.Xt_MVM=0;                       % number of carriers per unit of membrane area

param.KA1_MVM=4e3;                    % [micro mol/dm^3] dissociation constant side 1
param.KB1_MVM=param.KA1_MVM;          % same parameters used for solute B
param.KA2_MVM=param.KA1_MVM;          % [micro mol/dm^3] dissociation constant side 2
param.KB2_MVM=param.KA2_MVM;          % same parameters used for solute B
param.KK1_MVM=100/1e7*1e6;            % [uM] potassium dissociation constant side 1
param.KK2_MVM=param.KK1_MVM;          % [uM] potassium dissociation constant side 2
param.KNa1_MVM=1500/1e6*1e6;          % [uM] sodium dissociation constant side 1
param.KNa2A_MVM=param.KNa1_MVM;       % [uM] sodium dissociation constant side 2
param.KNa2B_MVM=param.KNa2A_MVM;      % same parameters used for solute B
param.kANaX12_MVM=100*60;             % [min^-1] forward rate constant
param.kBNaX12_MVM=param.kANaX12_MVM;  % same parameters used for solute B
param.kANaX21_MVM=param.kANaX12_MVM;  % [min^-1] reverse rate constant
param.kBNaX21_MVM=param.kANaX21_MVM;  % same parameters used for solute B
param.kKX12_MVM=3*60;                 % [min^-1]forward rate constant
param.kKX21_MVM=param.kKX12_MVM;      % [min^-1] reverse rate constant


% --- Efflux transporter on MVM:
param.VefMVM=0;                       % [micro mol/min] Vmax efflux transporter

param.KmefMVM=0.8;                    % [micro mol/dm^3] Km efflux transporter 


% --- OCT3 on BM: 
param.Xt_BM= 0;                       % number of carriers per unit of membrane area

param.kX12_BM= 1;                     % [min^-1] forward rate constant unloaded transporter (set to 1 as transport depends on product Xt and k)
param.kX21_BM=param.kX12_BM;          % [min^-1] reverse rate constant unloaded transporter
deltaPsi_BM=deltaPsi;                 % membrane potential difference, side 1 is inside the cell, side 2 outside 

% BM substrate A: Labeled metformin tracer
kA12_BM=10;                           % [min^-1] forward rate constant loaded transporter (relative to k for the unloaded transporter)
kA21_BM=kA12_BM;                      % [min^-1] reverse rate constant loaded transporter
param.kA12_BM=kA12_BM*exp(beta*z*F*deltaPsi_BM/(R*T));     % [min^-1] forward rate constant
param.kA21_BM=kA21_BM*exp((beta-1)*z*F*deltaPsi_BM/(R*T)); % [min^-1] reverse rate constant
param.KA1_BM=2.6e3;                                        % [micro mol/dm^3] dissociation constant side 1
param.KA2_BM=param.KA1_BM;                                 % [micro mol/dm^3] dissociation constant side 2

% BM substrate B: Unlabelled metformin, same properties as A
param.kB12_BM=param.kA12_BM;          % [min^-1] forward rate constant
param.kB21_BM=param.kA21_BM;          % [min^-1] reverse rate constant
param.KB1_BM=param.KA1_BM;            % [micro mol/dm^3] dissociation constant side 1
param.KB2_BM=param.KB1_BM;            % [micro mol/dm^3] dissociation constant side 2

% BM substrate C: Additional competitive inhibitor, same properties as A
param.kC12_BM=param.kA12_BM;          % [min^-1] forward rate constant
param.kC21_BM=param.kA21_BM;          % [min^-1] reverse rate constant
param.KC1_BM=param.KA1_BM;            % [micro mol/dm^3] dissociation constant side 1
param.KC2_BM=param.KC1_BM;            % [micro mol/dm^3] dissociation constant side 2


% Initial maternal concentrations
y1_0(1)=0;  % [micro mol/dm^3] initial external concentration substrate A: metformin tracer
y1_0(2)=0;  % [micro mol/dm^3] initial external concentration substrate B: unlabelled metformin
y1_0(3)=0;  % [micro mol/dm^3] initial external concentration substrate C: competitive inhibitor

% Initial internal syncytiotrophoblast concentrations     
y2_0(1)=0;    % [uM] initial concentration A 
y2_0(2)=0;    % [uM] initial concentration B 
y2_0(3)=0;    % [uM] initial concentration C 

% Initial fetal concentrations     
y3_0(1)=0;    % [uM] initial concentration A 
y3_0(2)=0;    % [uM] initial concentration B 
y3_0(3)=0;    % [uM] initial concentration C 

% Maternal inlet concentrations
param.Cinm=Cinm;

% % Fetal inlet concentrations
param.Cinf=Cinf;


Tstart=30; % [min] Start of actual experiments after setup phase
switch param.perfcase
    case 1
        Tend=150;    % [min] end time simulation Bidirectional experiments 
    case 2
        Tend=190;    % [min] end time simulation OCT3 experiments 
end
tspan=[Tstart,Tend];


% --- Initialize input parameters -----------------------------------------
naa=length(y1_0);     % number of substrates
ind1=1:naa;           % index for substrates in external compartment 1
ind2=naa+1:2*naa;     % index for substrates in internal compartment 2
ind3=2*naa+1:3*naa;   % index for substrates in internal compartment 3

param.naa=naa;
param.ind1=ind1;
param.ind2=ind2;
param.ind3=ind3;

% --- Compute results, time integration ---
y0 = [y1_0';y2_0';y3_0'];


if fitpar    % Fit parameters if fitpar = true, else use parameters given
    estpar0=[]; % Initial estimate of parameters to be fitted, using values specified below (vary these to confirm results)
    
    param.PdifA=0.0007;                      % diffusive permeability substrate A
    ubPdifA=1e-3;                            % upper bound used for fitting if using globalsearch (use varying range)

    param.Xt_MVM=0;                          % number of carriers per unit of membrane area
    ubXt_MVM=1e-3;                           % upper bound used for fitting if using globalsearch (use varying range)

    param.Xt_BM= 0;                          % number of carriers per unit of membrane area
    ubXt_BM=50;                              % upper bound used for fitting if using globalsearch (use varying range)
    
    param.VefMVM=0;                          % [micro mol/min] Vmax efflux transporter
    ubVefMVM=1e-3;                           % upper bound used for fitting if using globalsearch (use varying range)

    switch param.fitcase % 1=diffusion only, 2=diffusion+OCT3, 3=transporters only SERT, OCT3 and MVM efflux, 4=everything,diffusion and all transporters 
        case 1 %= diffusion only, set other parameters to zero
            estpar0(1)=param.PdifA;          % diffusive permeability substrate A
            param.Xt_BM=0;                   % OCT3 transporter on BM
            param.Xt_MVM=0;                  % uptake via SERT on MVM
            param.VefMVM=0;                  % efflux transporter on MVM
            
            %scalefac=estpar0;
            scalefac=ones(size(estpar0));
            normpar0=estpar0./scalefac;      % normalize parameters for fitting (set to one, so not used now)                       
            lb=zeros(size(normpar0));        % lower bound         
            ub=ubPdifA;                      % upper bound
            
            f=@(x)fminTWfunc_crea(x,scalefac,param,tspan,y0); 
            %[normparfit,fval]=fmincon(f,normpar0,[],[],[],[],lb,ub);   
            %[normparfit,fval]=fminsearch(f,normpar0);
            gs = GlobalSearch;               % slower, but more comprehensive search
            problem = createOptimProblem('fmincon','x0',normpar0,'objective',f,'lb',lb,'ub',ub);
            [normparfit,fval,exitflag,output,solutions] = run(gs,problem);                       
            
            estpar = normparfit.*scalefac;   % Scale back estimated parameters            
            % Replace parameters by estimates
            param.PdifA=estpar(1);           % [dm min^-1] diffusive permeability A 
            param.PdifB=param.PdifA ;        % [dm min^-1] diffusive permeability B
            param.PdifC=param.PdifA ;        % [dm min^-1] diffusive permeability C
            
            
        case 2  %= diffusion+OCT3,
            estpar0(1)=param.PdifA;          % diffusive permeability substrate A
            estpar0(2)=param.Xt_BM;          % OCT3 transporters on BM
            param.Xt_MVM=0;                  % uptake via SERT on MVM
            param.VefMVM=0;                  % efflux transporter on MVM
            
            %scalefac=estpar0;
            scalefac=ones(size(estpar0));
            normpar0=estpar0./scalefac;      % normalize parameters for fitting (set to one, so not used now)                                    
            lb=zeros(size(normpar0));        % lower bound         
            ub=[ubPdifA, ubXt_BM];           % upper bound
            
            f=@(x)fminTWfunc_crea(x,scalefac,param,tspan,y0); 
            %[normparfit,fval]=fmincon(f,normpar0,[],[],[],[],lb,ub);
            %[normparfit,fval]=fminsearch(f,normpar0);
            gs = GlobalSearch;               % slower, but more comprehensive search
            problem = createOptimProblem('fmincon','x0',normpar0,'objective',f,'lb',lb,'ub',ub);
            [normparfit,fval,exitflag,output,solutions] = run(gs,problem);
            
            estpar = normparfit.*scalefac;   % Scale back estimated parameters
            % Replace parameters by estimates
            param.PdifA=estpar(1);           % [dm min^-1] diffusive permeability A
            param.PdifB=param.PdifA ;        % [dm min^-1] diffusive permeability B
            param.PdifC=param.PdifA ;        % [dm min^-1] diffusive permeability C
            param.Xt_BM=estpar(2);           % OCT3 transporters on BM
                               
            
        case 3 %= transporters only SERT, OCT3 and MVM efflux
            param.PdifA=0;                   % [dm min^-1] diffusive permeability A 
            param.PdifB=param.PdifA ;        % [dm min^-1] diffusive permeability B
            param.PdifC=param.PdifA ;        % [dm min^-1] diffusive permeability C
            estpar0(1)=param.Xt_BM;          % OCT3 transporters on BM
            estpar0(2)=param.Xt_MVM;         % SERT transporters on MVM
            estpar0(3)=param.VefMVM;         % efflux transporter on MVM
            
            %scalefac=estpar0;
            scalefac=ones(size(estpar0));
            normpar0=estpar0./scalefac;      % normalize parameters for fitting (set to one, so not used now)   
            lb=zeros(size(normpar0));        % lower bound
            ub=[ubXt_BM, ubXt_MVM, ubVefMVM];% upper bound
            
            f=@(x)fminTWfunc_crea(x,scalefac,param,tspan,y0); 
            %[normparfit,fval]=fmincon(f,normpar0,[],[],[],[],lb,ub);
            %[normparfit,fval]=fminsearch(f,normpar0);            
            gs = GlobalSearch;               % slower, but more comprehensive search
            problem = createOptimProblem('fmincon','x0',normpar0,'objective',f,'lb',lb,'ub',ub);
            [normparfit,fval,exitflag,output,solutions] = run(gs,problem);            
                     
            estpar = normparfit.*scalefac;   % Scale back estimated parameters
            % Replace parameters by estimates
            param.Xt_BM=estpar(1);           % OCT3 transporters on BM
            param.Xt_MVM=estpar(2);          % SERT transporters on MVM
            param.VefMVM=estpar(3);          % efflux transporter on MVM
            
            
        case 4 %= everything, diffusion and all transporters            
            estpar0(1)=param.PdifA;          % [dm min^-1] diffusive permeability A 
            estpar0(2)=param.Xt_BM;          % OCT3 transporters on BM
            estpar0(3)=param.Xt_MVM;         % SERT transporters on MVM
            estpar0(4)=param.VefMVM;         % efflux transporter on MVM
            
            %scalefac=estpar0; 
            scalefac=ones(size(estpar0));
            normpar0=estpar0./scalefac;      % normalize parameters for fitting (set to one, so not used now)  
            lb=zeros(size(normpar0));        % lower bound
            ub=[ubPdifA,ubXt_BM, ubXt_MVM, ubVefMVM]; % upper bound
           
            f=@(x)fminTWfunc_crea(x,scalefac,param,tspan,y0); 
            %[normparfit,fval]=fmincon(f,normpar0,[],[],[],[],lb,ub);
            %[normparfit,fval]=fminsearch(f,normpar0);
            gs = GlobalSearch;               % slower, but more comprehensive search
            problem = createOptimProblem('fmincon','x0',normpar0,'objective',f,'lb',lb,'ub',ub);
            [normparfit,fval,exitflag,output,solutions] = run(gs,problem);            
            
            estpar = normparfit.*scalefac;   % Scale back estimated parameters
            % Replace parameters by estimates
            param.PdifA=estpar(1);           % [dm min^-1] diffusive permeability A 
            param.PdifB=param.PdifA ;        % [dm min^-1] diffusive permeability B
            param.PdifC=param.PdifA ;        % [dm min^-1] diffusive permeability C
            param.Xt_BM=estpar(2);           % OCT3 transporters on BM
            param.Xt_MVM=estpar(3);          % SERT transporters on MVM
            param.VefMVM=estpar(4);          % efflux transporter on MVM
    end

  

    % Display result on screen
    disp(['DATAFILE: ',param.expdat])
    disp([' Pparacellular = ',num2str(param.PdifA),' [dm^3 min^-1]'])
    disp([' Xt_BM = ',num2str( param.Xt_BM),' [micro mol dm^-2]'])
    disp([' Xt_MVM = ',num2str( param.Xt_MVM),' [micro mol dm^-2]'])
    disp([' VefMVM = ',num2str( param.VefMVM),' [micro mol dm^-2 min^-1]'])
    disp(' ')
    disp([' Error = ',num2str(fval)])
end


%--------------------------------------------------------------------------
% Calculate and plot results with given parameters
%--------------------------------------------------------------------------
odeoptions=[];
[tout, yout]=ode45(@kinetics_perfusion_crea,tspan,y0,odeoptions,param);  



% --- PLOTTING ------------------------------------------------------------
y1=yout(:,ind1);
y2=yout(:,ind2);
y3=yout(:,ind3);


figure(1)
clf
subplot(3,1,1)
hold on
plot(tout,y1(:,1),'b')
plot(tout,y1(:,2),'r')
plot(tout,y1(:,3),'g')
xlabel('Time [min]')
ylabel('Concentration [-]')
title('Maternal compartment')
legend('CA1','CB1','CC1')

subplot(3,1,2)
hold on
plot(tout,y2(:,1),'b')
plot(tout,y2(:,2),'r')
plot(tout,y2(:,3),'g')
xlabel('Time [min]')
ylabel('Concentration [-]')
title('Syncytiotrophoblast compartment')

subplot(3,1,3)
hold on
plot(tout,y3(:,1),'b')
plot(tout,y3(:,2),'r')
plot(tout,y3(:,3),'g')
xlabel('Time [min]')
ylabel('Concentration [-]')
title('Fetal compartment')


% Plot experimental data 
if plotdata
 
    figure(2)    
    subplot(2,1,1)
    hold on
    h=plot(tout,100*y3(:,1)/Cinf,'g');  % plot fetal results
    set(h,'Linewidth',2)
    h=errorbar(tfetal,100*Cfetal/Cinf,100*sfetal/Cinf); % fetal results, time points, mean and standard deviation
    set(h,'color','g')
    set(h,'linestyle','none')
    set(h,'marker','.')
    set(h,'Markersize',18)
    title(['Fetal: ', param.expdat],'Interpreter', 'none')
    xlabel('Time [min]')
    ylabel('C fetal vein [% stock]')
    xlim([Tstart, Tend]);
    
    subplot(2,1,2)
    hold on
    h=plot(tout,100*y1(:,1)/Cinm,'g');  % plot maternal results
    set(h,'Linewidth',2)
    h=errorbar(tmaternal,100*Cmaternal/Cinm,100*smaternal/Cinm); % maternal results, time points, mean and standard deviation
    set(h,'color','g')
    set(h,'linestyle','none')
    set(h,'marker','.')
    set(h,'Markersize',18)
    xlabel('Time [min]')
    ylabel('C maternal vein [% stock]')
    title(['Maternal: ', param.expdat],'Interpreter', 'none') 
    xlim([Tstart, Tend]);
    
end

% save data and figure
IDstr=[param.expdat,'_case_',num2str(param.fitcase)];
save(IDstr);
saveas(gcf,IDstr);






